<?php

namespace App\Http\Controllers;

use App\Models\Prize;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Support\Facades\Log;

class PrizeController extends Controller implements HasMiddleware
{
    /**
     * 
     * Exception Middleware to some method
     */
    public static function middleware()
    {
        return [
            new Middleware("auth:sanctum", except: ['index'])
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $prizes = Prize::all();
        return response()->json($prizes);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
            ]);

            $prize = Prize::create($request->only(['name', 'description']));

            return response()->json($prize, 201);
        } catch (\Exception $e) {
            Log::error('Failed to store prize: ' . $e->getMessage());
            return response()->json(['message' => 'Failed to store prize'], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Prize $prize)
    {
        return response()->json($prize);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Prize $prize)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $prize->update($request->only(['name', 'description']));

        return response()->json($prize);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Prize $prize)
    {
        $prize->wheels()->delete();

        $prize->delete();
        return response()->json(['message' => 'Prize deleted successfully']);
    }

    /**
     * Get wheels associated with the prize.
     */
    public function wheels(Prize $prize)
    {
        $wheels = $prize->wheels;
        return response()->json($wheels);
    }
}
