<?php

namespace App\Http\Controllers;

use App\Models\Wheel;
use App\Models\Prize;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class WheelController extends Controller implements HasMiddleware
{
    /**
     * 
     * Exception Middleware to some method
     */
    public static function middleware()
    {
        return [
            new Middleware("auth:sanctum", except: ['index'])
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $wheels = Wheel::all();
        return response()->json($wheels);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'label' => 'required|string|max:255',
                'weight' => 'required|numeric',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
                'imageRadius' => 'required|numeric',
                'imageScale' => 'required|numeric',
            ]);

            $imagePath = "";
            if ($request->hasFile('image')) {
                $image = $request->file('image')->store('images', 'public');
                $imagePath = "storage/{$image}";
            }

            $prize = Prize::create([
                'name' => $request->label,
                'description' => 'Hadiah untuk ' . $request->label,
            ]);

            Wheel::create([
                'prize_id' => $prize->id,
                'label' => $prize->name,
                'weight' => $request->weight,
                'image' => $imagePath,
                'imageRadius' => $request->imageRadius,
                'imageScale' => $request->imageScale,
            ]);

            return response()->json([
                "success" => true,
                'imagePath' => $imagePath,
                "message" => "Update berhasil!!",
            ], 201);
        } catch (\Exception $e) {
            Log::error('Failed to store wheel: ' . $e->getMessage());
            return response()->json(['message' => 'Failed to store wheel', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Wheel $wheel)
    {
        Log::info('Request Method: ' . $request->method());
        Log::info('Request Headers: ', $request->headers->all());
        Log::info('Request Data: ', $request->all());

        $request->validate([
            'prize_id' => 'exists:prizes,id',
            'label' => 'sometimes|required|string|max:255',
            'weight' => 'required|numeric',
            'imageRadius' => 'required|numeric',
            'imageScale' => 'required|numeric',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('images', 'public');
            $wheel->image = "storage/{$imagePath}";
        }

        if ($request->has('prize_id')) {
            $prize = Prize::findOrFail($request->prize_id);

            $prize->name = $request->label;
            $prize->save();

            $wheel->label = $prize->name;
            $wheel->prize_id = $prize->id;
        }

        $wheel->fill($request->only(['weight', 'imageRadius', 'imageScale']));
        $wheel->save();

        return response()->json([
            "success" => true,
            "message" => "Update berhasil!!",
        ], 200);
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Wheel $wheel)
    {

        $prize = $wheel->prize;

        $wheel->prize_id = null;
        $wheel->save();

        try {
            $wheel->delete();

            if ($prize) {
                $prize->delete();
            }
        } catch (\Exception $e) {
            return response()->json(['error' => 'Gagal menghapus Wheel atau Prize'], 500);
        }

        return response()->json(['message' => 'Wheel dan Prize berhasil dihapus']);
    }
}