import React, {
    useEffect,
    useImperativeHandle,
    useRef,
    forwardRef,
} from "react";
import { Wheel } from "spin-wheel";
import OverlayImage from "../../../assets/images/overlay.png";
import LoadImages from "./LoadImages";
import SpinSound from "../../../assets/audio/spin-sound.mp3";

interface Item {
    label?: string;
    weight?: number;
    image?: HTMLImageElement | null;
    imageRadius?: number;
    imageRotation?: number;
    imageScale?: number;
}

interface EasingFunction {
    label: string;
    function: ((t: number) => number) | null;
}

interface WheelSpinProps {
    items: Item[];
    easingFunctions: EasingFunction[];
    winningItem: number;
    easingFunction: number;
    revolutions: number;
    spinDirection: number;
}

interface WheelSpinHandle {
    spinWheel: (winningItem: number) => void;
    stopWheel: () => void;
}

const WheelSpin = forwardRef<WheelSpinHandle, WheelSpinProps>(
    (
        {
            items,
            easingFunctions,
            winningItem,
            easingFunction,
            revolutions,
            spinDirection,
        },
        ref
    ) => {
        const wheelRef = useRef<Wheel | null>(null);
        const containerRef = useRef<HTMLDivElement | null>(null);
        const audioRef = useRef<HTMLAudioElement | null>(null);

        useImperativeHandle(ref, () => ({
            spinWheel: (winningItem: number) => {
                const easingFunc =
                    easingFunctions[easingFunction]?.function || null;

                if (audioRef.current) {
                    audioRef.current.currentTime = 0;
                    audioRef.current.play();
                }

                wheelRef.current?.spinToItem(
                    winningItem,
                    4000,
                    true,
                    revolutions,
                    spinDirection,
                    easingFunc
                );
            },
            stopWheel: () => {
                wheelRef.current?.stop();
            },
        }));

        useEffect(() => {
            const container = containerRef.current;

            if (!container) return;
            if (wheelRef.current) wheelRef.current.remove();

            const overlayImage = new Image();
            overlayImage.src = OverlayImage;

            const itemImages = items.map((item) => {
                if (item.image) {
                    const img = new Image();
                    img.src = item.image;
                    return {
                        ...item,
                        image: img,
                    };
                }
                return item;
            });

            LoadImages(itemImages.map((item) => item.image).filter(Boolean))
                .then(() => {
                    const wheel = new Wheel(container as HTMLElement, {
                        items: itemImages.map((item) => ({
                            label: item.label || "",
                            weight: item.weight || 1,
                            image: item.image || null,
                            imageRadius: item.imageRadius || 0.5,
                            imageRotation: item.imageRotation || 0,
                            imageScale: item.imageScale || 1,
                        })),
                        radius: 0.6,
                        itemLabelAlign: "center",
                        itemLabelRadius: 0.8,
                        itemLabelRadiusMax: 0.5,
                        itemLabelFontSizeMax: 50,
                        itemLabelRotation: 90,
                        itemLabelBaselineOffset: -0.13,
                        itemLabelFont: "Pragati Narrow",
                        itemBackgroundColors: ["#c7160c", "#fff"],
                        itemLabelColors: ["#fff", "#c7160c"],
                        rotationSpeed: 100,
                        rotationSpeedMax: 700,
                        rotationResistance: -70,
                        // offset: { x: 20, y: 20 },
                        overlayImage: overlayImage,
                        lineWidth: 0,
                    });

                    const context = container
                        .querySelector("canvas")
                        ?.getContext("2d");
                    if (context && overlayImage.complete) {
                        context.drawImage(overlayImage, 0, 0, 500, 500);
                    }

                    wheelRef.current = wheel;
                })
                .catch((error) => {
                    console.error("Error loading images:", error);
                });

            return () => {
                wheelRef.current?.stop();
            };
        }, [items]);

        return (
            <div>
                <audio ref={audioRef} src={SpinSound} preload="auto" />
                <div
                    ref={containerRef}
                    className="mt-[10rem] sm:mt-0 wheel pointer-events-none rounded-full w-[400px] h-[400px]  md:w-[550px] md:h-[550px]"
                ></div>
            </div>
        );
    }
);

export default WheelSpin;
