import React, { useEffect, useRef, useState, useCallback } from "react";
import WheelSpin from "../components/WheelSpin/WheelSpin";
import Style from "../../css/home.module.css";
import Marquee from "react-fast-marquee";
import { sinInOut } from "../components/WheelSpin/Easing";
import { HiExclamation } from "react-icons/hi";
import { Toast, Spinner, Modal, Table, Button, List } from "flowbite-react";
import ConfettiExplosion, { ConfettiProps } from "react-confetti-explosion";
import winSound from "../../assets/audio/win.mp3";
import axios from "axios";
import audio from "../../assets/audio/slotgaming88.mp3";
import logo from "../../assets/images/slotgaming88-logo.png";

interface Wheel {
    id: number;
    label: string;
    weight: number;
    image: string;
    imageRadius: number;
    imageScale: number;
    created_at: string | null;
    updated_at: string | null;
}

interface Winner {
    username: string | undefined;
    prize: {
        name: string;
        description: string;
    };
}

const App: React.FC = () => {
    const wheelRef = useRef<any>(null);
    const [wheels, setWheels] = useState<Wheel[]>([]);
    const usernameRef = useRef<HTMLInputElement>(null);
    const ticketCodeRef = useRef<HTMLInputElement>(null);
    const [message, setMessage] = useState<string>("");
    const [winners, setWinners] = useState<Winner[]>([]);
    const [toastWinner, setToastWinner] = useState<boolean>(false);
    const [messageWinner, setMeesageWinner] = useState<string>("");
    const [isLoading, setIsLoading] = useState<boolean>(false);
    const [openModal, setOpenModal] = useState<boolean>(false);
    const [openingModal, setOpeningModal] = useState<boolean>(true);
    const [showConfetti, setShowConfetti] = useState<boolean>(false);
    const [prizeModal, setPrizeModal] = useState<boolean>(false);
    const [loadingPrize, setLoadingPrize] = useState<boolean>(false);
    const [prizes, setPrizes] = useState<any[]>([]);

    const audioRef = useRef<HTMLAudioElement | null>(null);
    const winAudioRef = useRef<HTMLAudioElement | null>(null);

    const largeProps: ConfettiProps = {
        force: 0.8,
        duration: 3000,
        particleCount: 300,
        width: 1600,
        colors: ["#041E43", "#1471BF", "#5BB4DC", "#FC027B", "#66D805"],
    };

    const playAudio = useCallback(() => {
        if (!audioRef.current) {
            audioRef.current = new Audio(audio);
        }

        audioRef.current.play().catch((error) => {
            console.error("Error playing audio:", error);
        });
    }, []);

    const playWinAudio = useCallback(() => {
        if (!winAudioRef.current) {
            winAudioRef.current = new Audio(winSound);
        }

        winAudioRef.current.play().catch((error) => {
            console.error("Error playing selamat audio:", error);
        });
    }, []);

    useEffect(() => {
        document.body.classList.add(Style.background);

        return () => {
            document.body.classList.remove(Style.background);
        };
    }, []);

    const fetchWinners = async () => {
        try {
            const response = await axios.get("/api/winners");
            setWinners(response.data.winners);
        } catch (error) {
            console.error(`Failed to fetch winners: ${error}`);
        }
    };

    const fetchPrize = async () => {
        setLoadingPrize(true);

        try {
            const response = await axios.get("/api/prize");
            setPrizes(response.data);
        } catch (error) {
            console.error("fetching prize error: ", error);
        } finally {
            setLoadingPrize(false);
        }
    };

    useEffect(() => {
        const interval = setInterval(fetchWinners, 60000);

        return () => clearInterval(interval);
    }, [winners]);

    const items = React.useMemo(() => {
        return wheels.map(({ id, created_at, updated_at, ...item }) => item);
    }, [wheels]);

    const defaultWinningItem = 3;
    const defaultEasingFunction = sinInOut;
    const defaultRevolutions = 50;
    const defaultSpinDirection = 1;

    const handleSpin = useCallback(
        async (event: React.FormEvent) => {
            event.preventDefault();

            setIsLoading(true);

            const username = usernameRef.current?.value;
            const ticketCode = ticketCodeRef.current?.value;

            if (wheelRef.current) {
                try {
                    const response = await axios.post("/api/redeem", {
                        username,
                        code: ticketCode,
                    });

                    setWinners((prevWinners: Winner[]) => [
                        ...prevWinners,
                        { username: username, prize: response.data.prize },
                    ]);

                    const winningIndex = items.findIndex(
                        (item) => item.label === response.data.prize.name
                    );

                    if (winningIndex !== -1 && !isLoading) {
                        wheelRef.current.spinWheel(winningIndex);

                        setTimeout(() => {
                            setToastWinner(true);
                            setMeesageWinner(
                                `TIKET BERHASIL DI REDEEM! ANDA MEMENAGKAN ${response.data.prize.name}`
                            );
                            setShowConfetti(true);
                            playWinAudio();
                            setTimeout(() => {
                                setShowConfetti(false);
                            }, 5000);
                        }, 4000);
                    }
                } catch (error) {
                    setMessage("TIKET TIDAK VALID ATAU SUDAH DI REDEEM");
                } finally {
                    setIsLoading(false);
                }
            }
        },
        [items]
    );

    useEffect(() => {
        if (message) {
            const timer = setTimeout(() => {
                setMessage("");
            }, 5000);

            return () => clearTimeout(timer);
        }
    }, [message]);

    useEffect(() => {
        const fetchWheel = async () => {
            try {
                const response = await axios.get("/api/wheels");
                setWheels(response.data);
            } catch (error) {
                console.error(`Failed to fetch wheels: ${error}`);
            }
        };

        fetchWheel();
    }, []);

    useEffect(() => {
        if (toastWinner && messageWinner) {
            setMessage("");
        }
        if (message) {
            setToastWinner(false);
        }
    }, [toastWinner, messageWinner, message]);

    return (
        <>
            {toastWinner && messageWinner && (
                <div className="fixed top-1 left-0 w-full flex justify-center z-50">
                    <Toast className="w-full max-w-md bg-green-500/70 text-white border border-green-600 backdrop-blur-sm shadow-lg rounded-xl">
                        <span className="h-5 w-5 mb-1">🎉</span>
                        <div className="ml-3 text-sm font-normal">
                            {messageWinner}
                        </div>
                        <Toast.Toggle
                            className="bg-transparent text-white"
                            onClick={() => {
                                setMeesageWinner("");
                                setToastWinner(false);
                            }}
                        />
                    </Toast>
                </div>
            )}
            <Modal
                show={openingModal}
                size="lg"
                position="center"
                onClose={() => {
                    playAudio();
                    setOpeningModal(false);
                }}
            >
                <Modal.Header className="border-0">
                    LUCKY SPIN WHEEL
                </Modal.Header>
                <Modal.Body className=" text-gray-500">
                    <h1 className="text-lg">SYARAT DAN KETENTUAN :</h1>
                    <List ordered>
                        <List.Item>
                            KUPON TIKET LUCKY SPIN HANYA DIKLAIM SETIAP AWAL
                            BULAN
                        </List.Item>
                        <List.Item>
                            EVENT INI HANYA BISA DI KLAIM MELALUI WHATSAPP &
                            LIVECHAT
                        </List.Item>
                        <List.Item>
                            UNTUK MENDAPATKAN EVENT INI MINIMAL TURNOVER DALAM
                            SEBULAN ADALAH <b>RP.100.000.000</b>
                        </List.Item>
                        <List.Item>
                            HADIAH UTAMA <b>IPHONE 16</b>, TANPA ADA ZONK
                        </List.Item>
                        <List.Item>
                            TURNOVER HANYA DIHITUNG DARI PERMAINAN SLOT & CASINO
                        </List.Item>
                        <List.Item>
                            BONUS LUCKY SPIN DI KLAIM SETIAP TANGGAL 1 PUKUL
                            15.00 WIB
                        </List.Item>
                        <List.Item>
                            TIDAK ADA SYARAT WITHDRAW DARI PEMENANG LUCKY SPIN
                            WHEEL
                        </List.Item>
                        <List.Item>
                            JIKA ADA TINDAKAN KECURANGAN DALAM PROMOSI INI,
                            PIHAK SLOTGAMING88 <b>BERHAK MEMBATALKAN</b> HADIAH
                            SPIN WHEEL
                        </List.Item>
                    </List>
                </Modal.Body>
                <Modal.Footer className="flex justify-center items-center border-0">
                    <Button
                        onClick={() => {
                            setOpeningModal(false);
                            playAudio();
                        }}
                        pill
                        className="w-full"
                    >
                        <h1 className="text-xl uppercase font-bold">MULAI</h1>
                    </Button>
                </Modal.Footer>
            </Modal>
            <div className="flex flex-col justify-center items-center h-screen p-5 my-[5rem] mt-[7rem] md:my-0">
                {showConfetti && <ConfettiExplosion {...largeProps} />}
                <div className="grid grid-cols-1 sm:grid-cols-2 gap-10 w-full max-w-6xl">
                    <div className="flex justify-center items-center">
                        <WheelSpin
                            ref={wheelRef}
                            items={items}
                            easingFunctions={[
                                {
                                    label: "easeSinOut",
                                    function: defaultEasingFunction,
                                },
                            ]}
                            winningItem={defaultWinningItem}
                            easingFunction={0}
                            revolutions={defaultRevolutions}
                            spinDirection={defaultSpinDirection}
                        />
                    </div>

                    <div className="flex flex-col justify-center items-center space-y-2">
                        <div>
                            <img
                                src={logo}
                                className="animate-bounce h-32"
                                alt="Logo"
                            />
                        </div>
                        <div
                            onClick={() => setOpenModal(true)}
                            className="w-full bg-gradient-to-b from-red-500 to-red-700 bg-opacity-20 opacity-90 rounded-2xl text-white py-2 text-center overflow-hidden hover:cursor-pointer"
                        >
                            <Marquee pauseOnHover gradient={false} speed={50}>
                                {winners.length > 0 ? (
                                    winners.map((winner, index) => {
                                        const maskedUsername =
                                            winner.username &&
                                            winner.username.length > 4
                                                ? winner.username.slice(0, 4) +
                                                  "*".repeat(
                                                      winner.username.length - 4
                                                  )
                                                : winner.username;
                                        return (
                                            <span key={index} className="mx-4">
                                                🎉 {maskedUsername} MEMENANGKAN{" "}
                                                {winner.prize?.name}! 🎉
                                            </span>
                                        );
                                    })
                                ) : (
                                    <span>BELUM ADA PEMENANG SAAT INI</span>
                                )}
                            </Marquee>
                        </div>

                        <div className="flex flex-col bg-gradient-to-b from-red-500 to-red-700 text-white p-3 rounded-3xl w-full px-5">
                            <h2 className="text-xl font-sm sm:font-sm text-center">
                                SELAMAT DATANG DI RODA HOKI{" "}
                                <b>SLOTGAMING88. </b>
                                SILAHKAN MASUKAN USERNAME DAN KODE TIKET YANG
                                TELAH ANDA DAPATKAN
                            </h2>
                        </div>
                        <form
                            className="flex flex-col space-y-2 w-full"
                            onSubmit={handleSpin}
                        >
                            <input
                                type="text"
                                placeholder="Username"
                                ref={usernameRef}
                                className="p-3 border-2 border-red-700 rounded-full w-full text-center focus:ring-1 text-gray-900 focus:ring-red-700 outline-none placeholder:text-lg"
                                required
                            />
                            <input
                                type="text"
                                placeholder="Kode Tiket"
                                ref={ticketCodeRef}
                                className="p-3 border-2 border-red-700 rounded-full w-full text-center focus:ring-1 text-gray-900 focus:ring-red-700 outline-none"
                                required
                            />
                            <div className="flex flex-col sm:flex-row gap-2 w-full">
                                <button
                                    type="submit"
                                    className="bg-gradient-to-b from-red-500 to-red-700 text-white p-3 rounded-full hover:from-red-600 hover:to-red-800 w-full focus:ring focus:ring-red-300 uppercase font-bold"
                                >
                                    {isLoading ? (
                                        <>
                                            <Spinner
                                                aria-label="Loading"
                                                size="sm"
                                            />{" "}
                                            <span className="ml-2">
                                                Loading...
                                            </span>
                                        </>
                                    ) : (
                                        "Redeem"
                                    )}
                                </button>
                            </div>

                            <button
                                onClick={() => {
                                    setPrizeModal(true);
                                    fetchPrize();
                                }}
                                type="button"
                                className="bg-gradient-to-b from-yellow-500 to-yellow-700 text-white p-3 rounded-full hover:from-yellow-600 hover:to-yellow-800 w-full focus:ring focus:ring-yellow-300 uppercase font-bold"
                            >
                                Hadiah
                            </button>
                        </form>

                        {message && (
                            <div className="fixed top-1 left-0 w-full flex justify-center z-50">
                                <Toast className="w-full max-w-md bg-red-500/70 backdrop-blur-sm text-white border border-red-600 shadow-lg rounded-xl">
                                    <div className="inline-flex h-8 w-8 shrink-0 items-center justify-center rounded-lg bg-orange-100 text-orange-500 dark:bg-orange-700 dark:text-orange-200">
                                        <HiExclamation className="h-5 w-5" />
                                    </div>
                                    <div className="ml-3 text-sm font-normal">
                                        {message}
                                    </div>
                                    <Toast.Toggle
                                        onClick={() => setMessage("")}
                                    />
                                </Toast>
                            </div>
                        )}
                    </div>
                </div>

                <Modal
                    show={prizeModal}
                    size="md"
                    onClose={() => setPrizeModal(false)}
                >
                    <Modal.Header className="border-0">
                        Daftar Hadiah
                    </Modal.Header>
                    <Modal.Body>
                        <Table>
                            <Table.Body className="p-0">
                                {loadingPrize ? (
                                    <Table.Row>
                                        <Table.Cell
                                            colSpan={2}
                                            className="text-center"
                                        >
                                            <Spinner size="xl" />
                                        </Table.Cell>
                                    </Table.Row>
                                ) : (
                                    prizes.map((prize, index) => (
                                        <Table.Row key={index}>
                                            <Table.Cell>
                                                {prize.name}
                                            </Table.Cell>
                                            <Table.Cell>
                                                {prize.description}
                                            </Table.Cell>
                                        </Table.Row>
                                    ))
                                )}
                            </Table.Body>
                        </Table>
                    </Modal.Body>
                </Modal>
                <Modal
                    show={openModal}
                    size="sm"
                    onClose={() => setOpenModal(false)}
                >
                    <Modal.Header>Daftar Riwayat Spin</Modal.Header>
                    <Modal.Body className="p-0">
                        <Table>
                            <Table.Head>
                                <Table.HeadCell className="w-0">
                                    NO
                                </Table.HeadCell>
                                <Table.HeadCell>USERNAME</Table.HeadCell>
                                <Table.HeadCell>WINNER</Table.HeadCell>
                            </Table.Head>
                            <Table.Body>
                                {winners.length === 0 && (
                                    <Table.Row>
                                        <Table.Cell
                                            colSpan={3}
                                            className="text-center"
                                        >
                                            Belum ada pemenang{" "}
                                        </Table.Cell>
                                    </Table.Row>
                                )}
                                {winners.map((winner, index) => (
                                    <Table.Row key={index}>
                                        <Table.Cell className="py-3">
                                            {index + 1}
                                        </Table.Cell>
                                        <Table.Cell className="py-3">
                                            {winner.username?.slice(0, 4) +
                                                "*".repeat(
                                                    winner.username?.length - 4
                                                )}
                                        </Table.Cell>
                                        <Table.Cell className="py-3">
                                            {winner.prize?.name}
                                        </Table.Cell>
                                    </Table.Row>
                                ))}
                            </Table.Body>
                        </Table>
                    </Modal.Body>
                </Modal>
            </div>
        </>
    );
};

export default App;
