import React from "react";
import { useDispatch, useSelector } from "react-redux";
import { useNavigate } from "react-router-dom";
import { RootState, AppDispatch } from "../store";
import { login } from "../store/authSlice";

interface Credentials {
    username: string;
    password: string;
}

const Login: React.FC = () => {
    const [username, setUsername] = React.useState<string>("");
    const [password, setPassword] = React.useState<string>("");

    const dispatch: AppDispatch = useDispatch();
    const navigate = useNavigate();
    const { loading, error, token } = useSelector(
        (state: RootState) => state.auth
    );

    const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
        e.preventDefault();

        const credentials: Credentials = {
            username,
            password,
        };

        await dispatch(login(credentials));
        console.log(error);
    };

    React.useEffect(() => {
        if (token) {
            navigate("/operator");
        }
    }, [token, navigate]);

    return (
        <div className="min-h-screen flex items-center justify-center bg-gradient-to-r from-red-600 to-purple-600 p-3">
            <div className="bg-white p-8 rounded-3xl shadow-lg max-w-md w-full space-y-6">
                <h2 className="text-2xl font-bold text-center">LOGIN PAGE</h2>

                {error && (
                    <div
                        className="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg relative"
                        role="alert"
                    >
                        <strong className="font-bold">Error! </strong>
                        <span className="block sm:inline">{error}</span>
                    </div>
                )}

                <form className="space-y-4" onSubmit={handleSubmit}>
                    <div>
                        <input
                            type="text"
                            className="mt-1 p-3 w-full border border-gray-400 rounded-full focus:ring-3 focus:ring-blue-300 focus:outline-none focus:border-blue-500"
                            placeholder="Username"
                            value={username}
                            onChange={(e) => setUsername(e.target.value)}
                            required
                        />
                    </div>
                    <div>
                        <input
                            type="password"
                            id="password"
                            className="mt-1 p-3 w-full border border-gray-400 rounded-full focus:ring-2 focus:ring-blue-300 focus:outline-none focus:border-blue-500"
                            placeholder="Password"
                            value={password}
                            onChange={(e) => setPassword(e.target.value)}
                            required
                        />
                    </div>
                    <div>
                        <button
                            type="submit"
                            className="w-full bg-blue-600 text-white p-3 rounded-full font-bold hover:bg-blue-700 focus:ring-2 focus:ring-blue-300 focus:outline-none transition uppercase"
                            disabled={loading}
                        >
                            {loading ? "Logging in..." : "Login"}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
};

export default Login;
