import React from "react";
import {
    Label,
    TextInput,
    Select,
    Button,
    Toast,
    Table,
    Spinner,
    Badge,
} from "flowbite-react";
import { useSelector } from "react-redux";
import { RootState } from "../../store";
import { BiSort } from "react-icons/bi";
import axios from "axios";
import { indexOf } from "lodash";

interface Prize {
    id: number;
    name: string;
    description: string;
}

interface Ticket {
    id: number;
    username: string;
    code: string;
    is_redeemed: number;
    prize: Prize;
}

interface Histories {
    id: number | null;
    username: string;
    code: string;
    prize: {
        name: string;
    };
    is_redeemed: number;
}

const Generate: React.FC = () => {
    const [prizes, setPrize] = React.useState<Prize[]>([]);
    const [tickets, setTickets] = React.useState<Ticket[]>([]);
    const [history, setHistory] = React.useState<Histories[]>([]);
    const [histories, setHistories] = React.useState<Histories[]>([]);
    const [message, setMessage] = React.useState<string>("");
    const [loadingTickets, setLoadingTickets] = React.useState<boolean>(true);
    const [username, setUsername] = React.useState<string>("");
    const [selectedPrize, setSelectedPrize] = React.useState<
        number | undefined
    >();
    const { token } = useSelector((state: RootState) => state.auth);
    const [sortField, setSortField] = React.useState<keyof Ticket | null>(null);
    const [sortOrder, setSortOrder] = React.useState<"asc" | "desc">("asc");

    const fetchPrize = async () => {
        try {
            const response = await axios.get<Prize[]>("/api/prize", {
                headers: {
                    Authorization: `Bearer ${token}`,
                },
            });
            setPrize(response.data);
        } catch (error) {
            console.error("fetching prize error: ", error);
        }
    };

    const fetchHistory = async () => {
        try {
            const response = await axios.get<Histories[]>("/api/history", {
                headers: {
                    Authorization: `Bearer ${token}`,
                },
            });
            setHistory(response.data);
            console.log(response.data);
        } catch (error) {
            console.log("fetching history error: ", error);
        }
    };

    const fetchTickets = async () => {
        try {
            const response = await axios.get<Ticket[]>("/api/ticket", {
                headers: {
                    Authorization: `Bearer ${token}`,
                },
            });
            setTickets(response.data);
        } catch (error) {
            console.error(`Failed to fetch tickets: ${error}`);
        } finally {
            setLoadingTickets(false);
        }
    };

    const handleSort = (field: keyof Ticket) => {
        const order =
            sortField === field && sortOrder === "asc" ? "desc" : "asc";
        setSortField(field);
        setSortOrder(order);
    };

    const sortedTickets = () => {
        const sorted = [...histories];
        if (sortField) {
            sorted.sort((a, b) => {
                const aValue = a[sortField];
                const bValue = b[sortField];

                if (typeof aValue === "string" && typeof bValue === "string") {
                    return sortOrder === "asc"
                        ? aValue.localeCompare(bValue)
                        : bValue.localeCompare(aValue);
                }

                return sortOrder === "asc"
                    ? (aValue as number) - (bValue as number)
                    : (bValue as number) - (aValue as number);
            });
        }
        return sorted;
    };

    const generateTicketCode = () => {
        const generatedCode = Array(8)
            .fill(0)
            .map(() => Math.random().toString(36).charAt(2))
            .join("")
            .toUpperCase();
        (document.getElementById("code") as HTMLInputElement).value =
            generatedCode;
    };

    const handleSave = async () => {
        if (!username || !selectedPrize) {
            setMessage("Please fill in all fields.");
            return;
        }

        try {
            await axios.post(
                "/api/ticket",
                {
                    username,
                    prize_id: selectedPrize,
                    code: (document.getElementById("code") as HTMLInputElement)
                        .value,
                },
                {
                    headers: {
                        Authorization: `Bearer ${token}`,
                    },
                }
            );

            setMessage("Ticket saved successfully!");
            fetchTickets();
        } catch (error) {
            console.error("Error saving ticket: ", error);
            setMessage("Error saving ticket.");
        }
    };

    const handleDelete = async (ticketId: number) => {
        try {
            await axios.delete(`/api/ticket/${ticketId}`, {
                headers: {
                    Authorization: `Bearer ${token}`,
                },
            });
            setMessage("Ticket deleted successfully!");
            fetchTickets();
        } catch (error) {
            console.error("Error deleting ticket: ", error);
            setMessage("Error deleting ticket.");
        }
    };

    React.useEffect(() => {
        const fetchData = async () => {
            await fetchPrize();
            await fetchTickets();
            await fetchHistory();
        };
        fetchData();
    }, []);

    React.useEffect(() => {
        setHistories(
            history.concat(tickets.filter((ticket) => ticket.is_redeemed === 0))
        );
    }, [history, tickets]);

    return (
        <div className="mb-4">
            {message && (
                <Toast className="fixed top-0 right-0 m-4 border border-blue-500 z-10">
                    <div className="ml-3 text-sm font-normal">{message}</div>
                    <Toast.Toggle onClick={() => setMessage("")} />
                </Toast>
            )}

            <div className="grid grid-cols-1 sm:grid-cols-1 md:grid-cols-3 gap-4">
                <div className="flex flex-col col-span-1 gap-4 bg-white border shadow-lg p-4 rounded-xl">
                    <h2 className="text-xl font-semibold mb-2">Generate</h2>
                    <Label htmlFor="username">Username</Label>
                    <TextInput
                        type="text"
                        id="username"
                        placeholder="johndoe"
                        required
                        value={username}
                        onChange={(e) => setUsername(e.target.value)}
                    />
                    <Label htmlFor="code">Kode Tiket</Label>
                    <div className="relative">
                        <TextInput
                            type="text"
                            id="code"
                            placeholder="Kode Ticket"
                            required
                            readOnly
                        />
                        <Button
                            size="xs"
                            className="absolute right-2 top-1/2 transform -translate-y-1/2 rounded-lg"
                            onClick={generateTicketCode}
                        >
                            Generate
                        </Button>
                    </div>
                    <Label htmlFor="prize">Prize</Label>
                    <Select
                        id="prize"
                        required
                        onChange={(e) =>
                            setSelectedPrize(Number(e.target.value))
                        }
                    >
                        <option value="">Select a prize</option>
                        {prizes.map((prize) => (
                            <option key={prize.id} value={prize.id}>
                                {prize.name}
                            </option>
                        ))}
                    </Select>
                    <div className="mt-3">
                        <Button
                            type="button"
                            className="font-bold uppercase w-full"
                            pill
                            onClick={handleSave}
                        >
                            Save
                        </Button>
                    </div>
                </div>

                <div className="col-span-2 overflow-x-auto bg-white border shadow-lg rounded-xl">
                    <h2 className="text-xl font-semibold mb-2 m-4">Tickets</h2>
                    <Table>
                        <Table.Head>
                            <Table.HeadCell
                                className="hover:cursor-pointer"
                                onClick={() => handleSort("id")}
                            >
                                <div className="flex justify-between">
                                    <span>No</span>
                                    <BiSort />
                                </div>
                            </Table.HeadCell>
                            <Table.HeadCell
                                className="hover:cursor-pointer"
                                onClick={() => handleSort("username")}
                            >
                                <div className="flex justify-between">
                                    <span>Username</span>
                                    <BiSort />
                                </div>
                            </Table.HeadCell>
                            <Table.HeadCell
                                className="hover:cursor-pointer"
                                onClick={() => handleSort("prize.name")}
                            >
                                <div className="flex justify-between">
                                    <span>Prize</span>
                                    <BiSort />
                                </div>
                            </Table.HeadCell>
                            <Table.HeadCell
                                className="hover:cursor-pointer"
                                onClick={() => handleSort("code")}
                            >
                                <div className="flex justify-between">
                                    <span>Code</span>
                                    <BiSort />
                                </div>
                            </Table.HeadCell>
                            <Table.HeadCell
                                className="hover:cursor-pointer"
                                onClick={() => handleSort("is_redeemed")}
                            >
                                <div className="flex justify-between">
                                    <span>Status</span>
                                    <BiSort />
                                </div>
                            </Table.HeadCell>
                            <Table.HeadCell>Action</Table.HeadCell>
                        </Table.Head>
                        <Table.Body className="divide-y">
                            {loadingTickets ? (
                                <Table.Row>
                                    <Table.Cell
                                        colSpan={6}
                                        className="text-center"
                                    >
                                        <Spinner />
                                    </Table.Cell>
                                </Table.Row>
                            ) : (
                                sortedTickets().map((ticket, index) => (
                                    <Table.Row key={ticket.id}>
                                        <Table.Cell>{index + 1}</Table.Cell>
                                        <Table.Cell>
                                            {ticket.username}
                                        </Table.Cell>
                                        <Table.Cell>
                                            {ticket.prize.name}
                                        </Table.Cell>
                                        <Table.Cell>{ticket.code}</Table.Cell>
                                        <Table.Cell>
                                            <Badge
                                                color={
                                                    ticket.is_redeemed === 1
                                                        ? "green"
                                                        : "red"
                                                }
                                            >
                                                {ticket.is_redeemed === 1
                                                    ? "Redeemed"
                                                    : "Not Redeemed"}
                                            </Badge>
                                        </Table.Cell>
                                        <Table.Cell>
                                            {!ticket.is_redeemed && (
                                                <Button
                                                    color="failure"
                                                    size={"xs"}
                                                    pill
                                                    onClick={() => {
                                                        if (
                                                            !ticket.is_redeemed
                                                        ) {
                                                            handleDelete(
                                                                ticket.id
                                                            );
                                                        }
                                                    }}
                                                >
                                                    Delete
                                                </Button>
                                            )}
                                        </Table.Cell>
                                    </Table.Row>
                                ))
                            )}
                        </Table.Body>
                    </Table>
                </div>
            </div>
        </div>
    );
};

export default Generate;
