import React from "react";
import axios from "axios";
import { useSelector } from "react-redux";
import { RootState } from "../../store";
import {
    Table,
    TableCell,
    TableHeadCell,
    TableRow,
    Button,
    Spinner,
    Modal,
    TextInput,
    Label,
    Alert,
} from "flowbite-react";

interface User {
    id: number;
    name: string;
    username: string;
    email: string;
}

const Settings: React.FC = () => {
    const [users, setUsers] = React.useState<User[]>([]);
    const [formData, setFormData] = React.useState({
        name: "",
        email: "",
        username: "",
        password: "",
        password_confirmation: "",
    });
    const [loading, setLoading] = React.useState<boolean>(true);
    const [openModal, setOpenModal] = React.useState(false);
    const [isEditing, setIsEditing] = React.useState(false);
    const [selectedUserId, setSelectedUserId] = React.useState<number | null>(
        null
    );
    const [errors, setErrors] = React.useState<string[]>([]);

    const { token } = useSelector((state: RootState) => state.auth);

    const handleChange = (e: any) => {
        const { name, value } = e.target;
        setFormData({
            ...formData,
            [name]: value,
        });
    };

    const fetchUsers = async () => {
        try {
            const response = await axios.get("/api/users", {
                headers: {
                    Authorization: `Bearer ${token}`,
                },
            });
            setUsers(response.data);
        } catch (error) {
            console.error("Fetching users failed: ", error);
        } finally {
            setLoading(false);
        }
    };

    React.useEffect(() => {
        fetchUsers();
    }, []);

    const handleSubmit = async () => {
        try {
            if (isEditing && selectedUserId) {
                await axios.put(`/api/users/${selectedUserId}`, formData, {
                    headers: {
                        Authorization: `Bearer ${token}`,
                    },
                });
            } else {
                await axios.post("/api/users", formData, {
                    headers: {
                        Authorization: `Bearer ${token}`,
                    },
                });
            }
            fetchUsers();
            setOpenModal(false);
            resetForm();
            setErrors([]);
        } catch (error: any) {
            if (error.response && error.response.data.errors) {
                setErrors(Object.values(error.response.data.errors).flat());
            } else {
                console.error("Submitting form failed: ", error);
            }
        }
    };

    const resetForm = () => {
        setFormData({
            name: "",
            email: "",
            username: "",
            password: "",
            password_confirmation: "",
        });
        setIsEditing(false);
        setSelectedUserId(null);
    };

    const handleEdit = (user: User) => {
        setFormData({
            name: user.name,
            email: user.email,
            username: user.username,
            password: "",
            password_confirmation: "",
        });
        setSelectedUserId(user.id);
        setIsEditing(true);
        setOpenModal(true);
    };

    const handleDelete = async (id: number) => {
        try {
            await axios.delete(`/api/users/${id}`, {
                headers: {
                    Authorization: `Bearer ${token}`,
                },
            });
            fetchUsers();
        } catch (error) {
            console.error("Deleting user failed: ", error);
        }
    };

    return (
        <div>
            <h2 className="text-xl font-semibold mb-4">Manage Users</h2>
            <Button size="sm" onClick={() => setOpenModal(true)}>
                Add User
            </Button>
            <div className="overflow-x-auto bg-white shadow-md rounded-xl my-5">
                <Table>
                    <Table.Head>
                        <TableHeadCell>No</TableHeadCell>
                        <TableHeadCell>Name</TableHeadCell>
                        <TableHeadCell>Username</TableHeadCell>
                        <TableHeadCell>Email</TableHeadCell>
                        <TableHeadCell>Action</TableHeadCell>
                    </Table.Head>
                    <Table.Body>
                        {loading && (
                            <TableRow className="text-center">
                                <TableCell colSpan={5}>
                                    <Spinner aria-label="Loading" />
                                </TableCell>
                            </TableRow>
                        )}
                        {users.map((user, index) => (
                            <TableRow key={index + 1}>
                                <TableCell>{index + 1}</TableCell>
                                <TableCell>{user.name}</TableCell>
                                <TableCell>{user.username}</TableCell>
                                <TableCell>{user.email}</TableCell>
                                <TableCell className="flex justify-center items-center gap-2">
                                    <Button
                                        size="xs"
                                        onClick={() => handleEdit(user)}
                                    >
                                        Edit
                                    </Button>
                                    <Button
                                        size="xs"
                                        color="failure"
                                        onClick={() => handleDelete(user.id)}
                                    >
                                        Delete
                                    </Button>
                                </TableCell>
                            </TableRow>
                        ))}
                    </Table.Body>
                </Table>
            </div>

            <Modal
                show={openModal}
                size="md"
                onClose={() => {
                    setOpenModal(false);
                    resetForm();
                    setErrors([]);
                }}
            >
                <Modal.Header className="border-0">
                    <h3 className="text-xl font-medium text-gray-900 dark:text-white">
                        {isEditing ? "Edit User" : "Add new user"}
                    </h3>
                </Modal.Header>
                <Modal.Body className="pt-0">
                    <div className="space-y-3">
                        <div>
                            <div className="mb-1 block">
                                <Label htmlFor="name" value="Fullname" />
                            </div>
                            <TextInput
                                type="text"
                                id="name"
                                name="name"
                                value={formData.name}
                                onChange={handleChange}
                                placeholder="John Doe"
                                required
                            />
                        </div>
                        <div>
                            <div className="mb-1 block">
                                <Label htmlFor="email" value="Email" />
                            </div>
                            <TextInput
                                type="email"
                                id="email"
                                name="email"
                                value={formData.email}
                                onChange={handleChange}
                                placeholder="johndoe@mail.com"
                                required
                            />
                        </div>
                        <div>
                            <div className="mb-1 block">
                                <Label htmlFor="username" value="Username" />
                            </div>
                            <TextInput
                                type="text"
                                id="username"
                                name="username"
                                value={formData.username}
                                onChange={handleChange}
                                placeholder="johndoe"
                                required
                            />
                        </div>
                        <div>
                            <div className="mb-1 block">
                                <Label htmlFor="password" value="Password" />
                            </div>
                            <TextInput
                                type="password"
                                id="password"
                                name="password"
                                value={formData.password}
                                onChange={handleChange}
                                placeholder="Password"
                                required={!isEditing}
                            />
                        </div>
                        <div>
                            <div className="mb-1 block">
                                <Label
                                    htmlFor="password_confirmation"
                                    value="Password Confirmation"
                                />
                            </div>
                            <TextInput
                                type="password"
                                id="password_confirmation"
                                name="password_confirmation"
                                value={formData.password_confirmation}
                                onChange={handleChange}
                                placeholder="Password Confirmation"
                                required={!isEditing}
                            />
                        </div>
                    </div>
                </Modal.Body>
                <Modal.Footer className="p-3 grid grid-cols-2 gap-2">
                    {errors.length > 0 && (
                        <Alert color="failure" className="col-span-2">
                            {errors.map((error, index) => (
                                <p key={index}>{error}</p>
                            ))}
                        </Alert>
                    )}
                    <Button color="failure" onClick={() => setOpenModal(false)}>
                        Close
                    </Button>
                    <Button onClick={handleSubmit}>
                        {isEditing ? "Update" : "Add"}
                    </Button>
                </Modal.Footer>
            </Modal>
        </div>
    );
};

export default Settings;
