import React, { useState, useEffect } from "react";
import { useSelector } from "react-redux";
import { RootState } from "../../store";
import {
    Table,
    Button,
    Modal,
    Label,
    TextInput,
    FileInput,
    Alert,
} from "flowbite-react";
import axios from "axios";

interface Wheel {
    id: number;
    prize_id: number;
    label: string;
    weight: number;
    image: string;
    imageRadius: number;
    imageScale: number;
}

interface Message {
    type: string;
    text: string;
}

const WheelSpin: React.FC = () => {
    const [wheels, setWheels] = useState<Wheel[]>([]);
    const [loading, setLoading] = useState<boolean>(true);
    const [error, setError] = useState<string | null>(null);
    const [wheelSelect, setWheelSelect] = useState<Wheel | null>(null);
    const [selectedImage, setSelectedImage] = useState<File | null>(null);

    const [message, setMessage] = useState<Message | null>({
        type: "",
        text: "",
    });

    const { token } = useSelector((state: RootState) => state.auth);
    const [openModal, setOpenModal] = useState<boolean>(false);
    const [openAddModal, setOpenAddModal] = useState<boolean>(false);

    const fetchWheels = async () => {
        try {
            const response = await axios.get("/api/wheels");
            setWheels(response.data);
        } catch (err) {
            setError("Failed to fetch wheels");
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchWheels();
    }, []);

    const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const { id, value } = e.target;
        if (wheelSelect) {
            setWheelSelect({
                ...wheelSelect,
                [id]: value,
            });
        }
    };

    const handleOpenAddModal = () => {
        setWheelSelect({
            id: 0,
            prize_id: 0,
            label: "",
            weight: 0,
            image: "",
            imageRadius: 0,
            imageScale: 0,
        });
        setSelectedImage(null);
        setOpenAddModal(true);
    };

    const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files && e.target.files[0]) {
            setSelectedImage(e.target.files[0]);
        }
    };

    const handleSave = async () => {
        if (!wheelSelect) return;

        if (
            !wheelSelect.label ||
            isNaN(wheelSelect.weight) ||
            isNaN(wheelSelect.imageRadius) ||
            isNaN(wheelSelect.imageScale)
        ) {
            alert("Please fill all fields correctly.");
            return;
        }

        try {
            const formData = new FormData();
            formData.append("prize_id", String(wheelSelect.prize_id));
            formData.append("label", wheelSelect.label);
            formData.append("weight", String(wheelSelect.weight));
            formData.append("imageRadius", String(wheelSelect.imageRadius));
            formData.append("imageScale", String(wheelSelect.imageScale));
            formData.append("_method", "PUT");

            if (selectedImage) {
                formData.append("image", selectedImage);
            }

            const response = await axios.post(
                `/api/wheels/${wheelSelect.id}`,
                formData,
                {
                    headers: {
                        Authorization: `Bearer ${token}`,
                        // "Content-Type": "multipart/form-data",
                    },
                }
            );
            setMessage({
                type: "success",
                text: response.data.message,
            });
            fetchWheels();
        } catch (error) {
            setMessage({
                type: "failed",
                text: "Failed to update wheel",
            });
            console.error("Failed to update wheel", error);
        }
    };

    const handleAdd = async () => {
        try {
            const formData = new FormData();
            formData.append("label", wheelSelect?.label || "");
            formData.append("weight", String(wheelSelect?.weight || 2));
            formData.append(
                "imageRadius",
                String(wheelSelect?.imageRadius || 0)
            );
            formData.append(
                "imageScale",
                String(wheelSelect?.imageScale || 0.13)
            );

            if (selectedImage) {
                formData.append("image", selectedImage);
            }

            console.log(formData);

            const response = await axios.post(`/api/wheels`, formData, {
                headers: {
                    Authorization: `Bearer ${token}`,
                    "Content-Type": "multipart/form-data",
                },
            });
            setMessage({
                type: "success",
                text: response.data.message,
            });
            fetchWheels();
        } catch (error) {
            setMessage({
                type: "failed",
                text: "Failed to update wheel",
            });
            console.error("Failed to add wheel", error);
        }
    };

    const handleDelete = async (id: number) => {
        const confirmDelete = window.confirm(
            "Are you sure you want to delete this wheel?"
        );
        if (!confirmDelete) return;

        try {
            await axios.delete(`/api/wheels/${id}`, {
                headers: {
                    Authorization: `Bearer ${token}`,
                },
            });
            fetchWheels();
        } catch (error) {
            console.error("Failed to delete wheel", error);
        }
    };

    if (loading) {
        return <p>Loading...</p>;
    }

    if (error) {
        return <p className="text-red-500">{error}</p>;
    }

    return (
        <div>
            <h2 className="text-xl font-semibold mb-4">Wheel Spin</h2>

            <Button
                onClick={handleOpenAddModal}
                pill
                size="sm"
                className="mb-4"
            >
                Add Wheel
            </Button>

            {wheels.length > 0 ? (
                <div className="overflow-x-auto">
                    <Table>
                        <Table.Head>
                            <Table.HeadCell>Label</Table.HeadCell>
                            <Table.HeadCell>Image</Table.HeadCell>
                            <Table.HeadCell className="text-center">
                                Action
                            </Table.HeadCell>
                        </Table.Head>
                        <Table.Body className="divide-y">
                            {wheels.map((wheel, index) => (
                                <Table.Row
                                    key={index}
                                    className="bg-white dark:border-gray-700 dark:bg-gray-800"
                                >
                                    <Table.Cell className="whitespace-nowrap font-medium text-gray-900 dark:text-white">
                                        {wheel.label}
                                    </Table.Cell>
                                    <Table.Cell className="whitespace-nowrap font-medium text-gray-900 dark:text-white">
                                        {wheel.image !== "0" ? (
                                            <img
                                                src={`/${wheel.image}`}
                                                alt={wheel.label}
                                                className="w-12 h-12"
                                            />
                                        ) : (
                                            <span className="italic text-xs text-gray-500">
                                                no image
                                            </span>
                                        )}
                                    </Table.Cell>
                                    <Table.Cell className="flex items-center justify-center gap-2 whitespace-nowrap font-medium text-gray-900 dark:text-white">
                                        <Button
                                            size="xs"
                                            onClick={() => {
                                                setOpenModal(true);
                                                setWheelSelect(wheel);
                                            }}
                                        >
                                            Edit
                                        </Button>
                                        <Button
                                            size="xs"
                                            color="failure"
                                            onClick={() =>
                                                handleDelete(wheel.id)
                                            }
                                        >
                                            Delete
                                        </Button>
                                    </Table.Cell>
                                </Table.Row>
                            ))}
                        </Table.Body>
                    </Table>
                </div>
            ) : (
                <p>No wheels available</p>
            )}

            <Modal
                show={openAddModal}
                size="md"
                onClose={() => {
                    setOpenAddModal(false);
                    setMessage({ type: "", text: "" });
                }}
            >
                <Modal.Header>Add New Wheel</Modal.Header>
                <Modal.Body>
                    <div>
                        <div className="mb-2">
                            <div className="mb-2 block">
                                <Label htmlFor="label" value="Label" />
                            </div>
                            <TextInput
                                id="label"
                                type="text"
                                placeholder="10K"
                                onChange={handleInputChange}
                                required
                                shadow
                            />
                        </div>
                        <div className="mb-2">
                            <div className="mb-2 block">
                                <Label htmlFor="weight" value="Weight" />
                            </div>
                            <TextInput
                                id="weight"
                                type="text"
                                placeholder="2.0"
                                defaultValue={"2.0"}
                                onChange={handleInputChange}
                                required
                                shadow
                            />
                        </div>
                        <div className="mb-2">
                            <div className="mb-2 block">
                                <Label htmlFor="image" value="Image" />
                            </div>
                            <FileInput
                                id="image"
                                helperText="SVG, PNG, JPG or GIF"
                                onChange={handleFileChange}
                            />
                        </div>
                        <div className="mb-2">
                            <div className="mb-2 block">
                                <Label
                                    htmlFor="imageRadius"
                                    value="Image Radius"
                                />
                            </div>
                            <TextInput
                                id="imageRadius"
                                type="number"
                                placeholder="0.0"
                                defaultValue={"0.0"}
                                onChange={handleInputChange}
                                required
                                shadow
                            />
                        </div>
                        <div className="mb-2">
                            <div className="mb-2 block">
                                <Label
                                    htmlFor="imageScale"
                                    value="Image Scale"
                                />
                            </div>
                            <TextInput
                                id="imageScale"
                                type="number"
                                placeholder="0.13"
                                defaultValue={"0.13"}
                                onChange={handleInputChange}
                                required
                                shadow
                            />
                        </div>
                    </div>
                </Modal.Body>
                <Modal.Footer className="grid grid-cols-2 gap-4">
                    <Button
                        color="failure"
                        onClick={() => {
                            setOpenAddModal(false);
                            setMessage({ type: "", text: "" });
                        }}
                    >
                        Cancel
                    </Button>
                    <Button onClick={handleAdd}>Add</Button>
                    {message?.text && (
                        <div className="col-span-2">
                            <Alert
                                color={
                                    message?.type === "success"
                                        ? "success"
                                        : "failure"
                                }
                                onDismiss={() =>
                                    setMessage({ type: "", text: "" })
                                }
                            >
                                {message?.text}
                            </Alert>
                        </div>
                    )}
                </Modal.Footer>
            </Modal>

            <Modal
                show={openModal}
                size="md"
                onClose={() => {
                    setOpenModal(false);
                    setMessage({ type: "", text: "" });
                }}
            >
                <Modal.Header>Update Wheel</Modal.Header>
                <Modal.Body>
                    {wheelSelect && (
                        <div>
                            <div className="mb-2">
                                <div className="mb-2 block">
                                    <Label htmlFor="label" value="Label" />
                                </div>
                                <TextInput
                                    id="label"
                                    type="text"
                                    value={wheelSelect.label}
                                    onChange={handleInputChange}
                                    required
                                    shadow
                                />
                            </div>
                            <div className="mb-2">
                                <div className="mb-2 block">
                                    <Label htmlFor="weight" value="Weight" />
                                </div>
                                <TextInput
                                    id="weight"
                                    type="text"
                                    value={wheelSelect.weight}
                                    onChange={handleInputChange}
                                    required
                                    shadow
                                />
                            </div>
                            <div className="mb-2">
                                <div className="mb-2 block">
                                    <Label htmlFor="image" value="Image" />
                                </div>
                                <FileInput
                                    id="image"
                                    helperText="SVG, PNG, JPG or GIF"
                                    onChange={handleFileChange}
                                />
                            </div>
                            <div className="mb-2">
                                <div className="mb-2 block">
                                    <Label
                                        htmlFor="imageRadius"
                                        value="Image Radius"
                                    />
                                </div>
                                <TextInput
                                    id="imageRadius"
                                    type="number"
                                    value={wheelSelect.imageRadius}
                                    onChange={handleInputChange}
                                    required
                                    shadow
                                />
                            </div>
                            <div className="mb-2">
                                <div className="mb-2 block">
                                    <Label
                                        htmlFor="imageScale"
                                        value="Image Scale"
                                    />
                                </div>
                                <TextInput
                                    id="imageScale"
                                    type="number"
                                    value={wheelSelect.imageScale}
                                    onChange={handleInputChange}
                                    required
                                    shadow
                                />
                            </div>
                        </div>
                    )}
                </Modal.Body>
                <Modal.Footer className="grid grid-cols-2 gap-4">
                    <Button
                        color="failure"
                        onClick={() => {
                            setOpenModal(false);
                            setMessage({ type: "", text: "" });
                        }}
                    >
                        Cancel
                    </Button>
                    <Button onClick={handleSave}>Save</Button>
                    {message?.text && (
                        <div className="col-span-2">
                            <Alert
                                color={
                                    message?.type === "success"
                                        ? "success"
                                        : "failure"
                                }
                            >
                                {message?.text}
                            </Alert>
                        </div>
                    )}
                </Modal.Footer>
            </Modal>
        </div>
    );
};

export default WheelSpin;
